<?php
/**
 * Quick Course Registration Script with RPMS Sync
 * Single page for student course registration with auto-insertion of core courses
 *
 * Process:
 * 1. Student enters matric number, selects semester, session, and level
 * 2. Fetches student records from OLD database
 * 3. Uses pcode, session, level, verif (must be "ACTIVE") to fetch curriculum
 * 4. Auto pre-inserts courses with status "C" (Core) to NEW database
 * 5. Enrolls student in RPMS and registers courses
 * 6. Records pulled from old have is_pushed = 1
 * 7. Final submit shows the course registration form (print view)
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);
date_default_timezone_set('Africa/Lagos');

// Get script directory for relative includes
$scriptDir = dirname(__FILE__);

// Include RPMS webserver functions
require_once($scriptDir . '/service/webserver.php');

// Database credentials
$old_db_config = [
    'host' => '162.214.100.197',
    'user' => 'OBanjo_m3',
    'pass' => 'jsA95d!7',
    'name' => 'OICT_mission'
];

$new_db_config = [
    'host' => '50.6.4.83',
    'user' => 'eportal_main_student',
    'pass' => '}NGP[4r]pXn+g_^5',
    'name' => 'eportal_main_student'
];

// RPMS API credentials
$rpmsUsername = 'eportal';
$rpmsPassword = 'Ntj)K&eH]waX';

// Allowed RPMS response codes (approved)
$allowedRpmsCodes = ['Z00', 'Z13', 'Z16'];

// Connect to databases
function connectDB($config) {
    $conn = new mysqli($config['host'], $config['user'], $config['pass'], $config['name']);
    if ($conn->connect_error) {
        die("Connection failed to {$config['host']}: " . $conn->connect_error);
    }
    $conn->set_charset('utf8');
    return $conn;
}

$old_db = null;
$new_db = null;
$message = '';
$messageType = '';
$student = null;
$curriculum = [];
$registeredCourses = [];
$showForm = true;
$showPrint = false;
$rpmsStatus = [];

// Process form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $regnum = trim($_POST['regnum'] ?? '');
    $session = trim($_POST['session'] ?? '');
    $semester = strtoupper(trim($_POST['semester'] ?? ''));
    $level = intval($_POST['level'] ?? 0);
    $action = $_POST['action'] ?? 'search';

    if (empty($regnum) || empty($session) || empty($semester) || empty($level)) {
        $message = 'All fields are required.';
        $messageType = 'error';
    } else {
        try {
            // Connect to old database
            $old_db = connectDB($old_db_config);

            // Fetch student from old database
            $stmt = $old_db->prepare("SELECT * FROM students WHERE regnum = ? OR jamb = ?");
            $stmt->bind_param("ss", $regnum, $regnum);
            $stmt->execute();
            $result = $stmt->get_result();
            $student = $result->fetch_assoc();
            $stmt->close();

            if (!$student) {
                $message = "Student with matric number '$regnum' not found.";
                $messageType = 'error';
            } elseif ($student['verif'] !== 'ACTIVE') {
                $message = "Student verification status is '{$student['verif']}'. Must be 'ACTIVE' to register courses.";
                $messageType = 'error';
                $student = null;
            } else {
                // Use ocode if available, otherwise pcode
                $pcode = (!empty($student['ocode'])) ? $student['ocode'] : $student['pcode'];

                // Special handling for NPE at levels 100 and 200
                if ($pcode === 'NPE' && in_array($level, [100, 200])) {
                    $pcode = 'ECH';
                }

                // Fetch curriculum from old database
                $currQuery = "SELECT * FROM curriculum WHERE pcode = ? AND session = ? AND level = ? AND semester = ? ORDER BY status ASC, code ASC";
                $stmt = $old_db->prepare($currQuery);
                $stmt->bind_param("ssis", $pcode, $session, $level, $semester);
                $stmt->execute();
                $currResult = $stmt->get_result();

                while ($row = $currResult->fetch_assoc()) {
                    $curriculum[] = $row;
                }
                $stmt->close();

                // If no curriculum found with exact pcode, try with student's pcode
                if (empty($curriculum) && $pcode !== $student['pcode']) {
                    $stmt = $old_db->prepare($currQuery);
                    $stmt->bind_param("ssis", $student['pcode'], $session, $level, $semester);
                    $stmt->execute();
                    $currResult = $stmt->get_result();
                    while ($row = $currResult->fetch_assoc()) {
                        $curriculum[] = $row;
                    }
                    $stmt->close();
                }

                // If still no curriculum, try without semester filter
                if (empty($curriculum)) {
                    $currQuery2 = "SELECT * FROM curriculum WHERE (pcode = ? OR pcode = ?) AND session = ? AND level = ? ORDER BY semester, status ASC, code ASC";
                    $stmt = $old_db->prepare($currQuery2);
                    $stmt->bind_param("ssis", $pcode, $student['pcode'], $session, $level);
                    $stmt->execute();
                    $currResult = $stmt->get_result();
                    while ($row = $currResult->fetch_assoc()) {
                        if (strtoupper($row['semester']) === $semester) {
                            $curriculum[] = $row;
                        }
                    }
                    $stmt->close();
                }

                if (empty($curriculum)) {
                    $message = "No curriculum found for Programme: $pcode, Session: $session, Level: $level, Semester: $semester";
                    $messageType = 'warning';
                } else {
                    $message = "Found " . count($curriculum) . " courses in curriculum. " . count(array_filter($curriculum, fn($c) => $c['status'] === 'C')) . " are core (C) courses.";
                    $messageType = 'success';
                }

                // If action is register, insert to new database and sync with RPMS
                if ($action === 'register' && !empty($curriculum)) {
                    $new_db = connectDB($new_db_config);

                    // Check if course_reg exists
                    $checkStmt = $new_db->prepare("SELECT sn FROM course_reg WHERE regnum = ? AND session = ? AND semester = ? AND level = ?");
                    $checkStmt->bind_param("ssis", $student['regnum'], $session, $semester, $level);
                    $checkStmt->execute();
                    $checkResult = $checkStmt->get_result();
                    $existingReg = $checkResult->fetch_assoc();
                    $checkStmt->close();

                    $regdate = date('Y-m-d H:i:s');

                    if ($existingReg) {
                        $creg_id = $existingReg['sn'];
                        // Update existing record
                        $updateStmt = $new_db->prepare("UPDATE course_reg SET regdate = ?, is_pushed = 1 WHERE sn = ?");
                        $updateStmt->bind_param("si", $regdate, $creg_id);
                        $updateStmt->execute();
                        $updateStmt->close();
                    } else {
                        // Insert new course_reg record
                        $insertStmt = $new_db->prepare("INSERT INTO course_reg (regnum, session, semester, level, regdate, stat, is_pushed) VALUES (?, ?, ?, ?, ?, 0, 1)");
                        $insertStmt->bind_param("sssis", $student['regnum'], $session, $semester, $level, $regdate);
                        $insertStmt->execute();
                        $creg_id = $new_db->insert_id;
                        $insertStmt->close();
                    }

                    // Insert core courses (status = 'C') into course_regs
                    $insertedCount = 0;
                    $totalUnits = 0;
                    $coreCourses = [];
                    $courseCodes = [];

                    foreach ($curriculum as $course) {
                        if ($course['status'] === 'C') {
                            // Check if already registered
                            $checkCourse = $new_db->prepare("SELECT id FROM course_regs WHERE creg_id = ? AND ccode = ?");
                            $checkCourse->bind_param("is", $creg_id, $course['code']);
                            $checkCourse->execute();
                            $courseExists = $checkCourse->get_result()->fetch_assoc();
                            $checkCourse->close();

                            if (!$courseExists) {
                                $insertCourse = $new_db->prepare("INSERT INTO course_regs (creg_id, ccode, course_id, score, rpms, is_pushed) VALUES (?, ?, ?, 0, 0, 1)");
                                $insertCourse->bind_param("isi", $creg_id, $course['code'], $course['sn']);
                                $insertCourse->execute();
                                $insertCourse->close();
                                $insertedCount++;
                            }
                            $totalUnits += intval($course['unit']);
                            $coreCourses[] = $course;
                            $courseCodes[] = $course['code'];
                        }
                    }

                    // Update total credits in course_reg
                    $updateTC = $new_db->prepare("UPDATE course_reg SET tc = ? WHERE sn = ?");
                    $updateTC->bind_param("ii", $totalUnits, $creg_id);
                    $updateTC->execute();
                    $updateTC->close();

                    // ============================================
                    // RPMS SYNC SECTION
                    // ============================================
                    $rpmsMessage = '';
                    $rpmsSuccess = false;

                    // Strip leading zeros for RPMS matric number
                    $matric = ltrim($student['regnum'], '0');
                    $courseCodesStr = implode(',', $courseCodes);

                    try {
                        // Step 1: Get RPMS API key
                        $rpmsKey = getKey($rpmsUsername, $rpmsPassword);

                        if ($rpmsKey && $rpmsKey !== 'Error' && $rpmsKey !== 'Fault' && strlen($rpmsKey) > 30) {

                            // Step 2: Check if student is enrolled in RPMS
                            $hasEnrollment = HasEnrollment($matric, $session, $semester, $rpmsKey);

                            // Handle invalid auth key - try to refresh
                            if ($hasEnrollment === 'Invalid Auth Key') {
                                $rpmsKey = getKey($rpmsUsername, $rpmsPassword);
                                if ($rpmsKey && strlen($rpmsKey) > 30) {
                                    $hasEnrollment = HasEnrollment($matric, $session, $semester, $rpmsKey);
                                }
                            }

                            // Step 3: Enroll student if not already enrolled
                            if ($hasEnrollment !== 'Yes') {
                                $enrollResult = enrollStudent($matric, $session, $semester, $pcode, $level, $rpmsKey);
                                if ($enrollResult === 'Sucessful' || $enrollResult === 'Successful') {
                                    $rpmsMessage .= "Student enrolled in RPMS. ";
                                }
                            } else {
                                $rpmsMessage .= "Student already enrolled in RPMS. ";
                            }

                            // Step 4: Register courses with RPMS
                            $registerResult = registerStudentCourses2($matric, $session, $semester, $courseCodesStr, $rpmsKey, true);

                            if (is_array($registerResult) && isset($registerResult['CourseRegistrationResponse'])) {
                                $allApproved = true;
                                $approvedCount = 0;
                                $rejectedCount = 0;
                                $syncedCourses = [];

                                foreach ($registerResult['CourseRegistrationResponse'] as $courseResponse) {
                                    $courseCode = $courseResponse['CourseCode'] ?? '';
                                    $responseCode = $courseResponse['ResponseCode'] ?? '';
                                    $responseDesc = $courseResponse['ResponseDescription'] ?? '';

                                    // Check if approved
                                    if (in_array($responseCode, $allowedRpmsCodes)) {
                                        $rpmsStatusVal = 1;
                                        $approvedCount++;
                                        $syncedCourses[] = $courseCode;
                                    } else {
                                        $rpmsStatusVal = 0;
                                        $allApproved = false;
                                        $rejectedCount++;
                                    }

                                    // Store status for display
                                    $rpmsStatus[$courseCode] = [
                                        'rpms' => $rpmsStatusVal,
                                        'remarks' => $responseDesc
                                    ];

                                    // Update course_regs with RPMS status
                                    $updateRpms = $new_db->prepare("UPDATE course_regs SET remarks = ?, rpms = ? WHERE ccode = ? AND creg_id = ?");
                                    $updateRpms->bind_param("sisi", $responseDesc, $rpmsStatusVal, $courseCode, $creg_id);
                                    $updateRpms->execute();
                                    $updateRpms->close();
                                }

                                // Update course_reg sync status
                                $syncStatus = $allApproved ? 'SUCCESSFUL' : 'UNSUCCESSFUL';
                                $syncComment = $allApproved ? 'All courses approved' : "Approved: $approvedCount, Rejected: $rejectedCount";
                                $syncedSerialized = serialize($syncedCourses);

                                $updateSync = $new_db->prepare("UPDATE course_reg SET stat = ?, sync = ?, syncom = ?, syncdate = ?, synced = ? WHERE sn = ?");
                                $statValue = $allApproved ? 1 : 0;
                                $updateSync->bind_param("issssi", $statValue, $syncStatus, $syncComment, $regdate, $syncedSerialized, $creg_id);
                                $updateSync->execute();
                                $updateSync->close();

                                $rpmsMessage .= "RPMS: $approvedCount approved, $rejectedCount rejected.";
                                $rpmsSuccess = $allApproved;
                            } else {
                                $rpmsMessage .= "RPMS returned unexpected response.";
                            }
                        } else {
                            $rpmsMessage = "Failed to get RPMS API key. Courses saved locally only.";
                        }
                    } catch (Exception $e) {
                        $rpmsMessage = "RPMS Error: " . $e->getMessage();
                    }

                    $message = "Registered $insertedCount core courses ($totalUnits units). $rpmsMessage";
                    $messageType = $rpmsSuccess ? 'success' : 'warning';
                    $showForm = false;
                    $showPrint = true;

                    // Fetch all registered courses for display
                    $fetchReg = $new_db->prepare("SELECT c.*, cr.rpms, cr.remarks FROM curriculum c INNER JOIN course_regs cr ON c.sn = cr.course_id WHERE cr.creg_id = ? ORDER BY cr.rpms DESC, c.status ASC, c.code ASC");
                    $fetchReg->bind_param("i", $creg_id);
                    $fetchReg->execute();
                    $regResult = $fetchReg->get_result();
                    $registeredCourses = [];
                    while ($row = $regResult->fetch_assoc()) {
                        $registeredCourses[] = $row;
                    }
                    $fetchReg->close();
                }
            }

        } catch (Exception $e) {
            $message = "Error: " . $e->getMessage();
            $messageType = 'error';
        }
    }
}

// Generate session options (last 10 years)
$currentYear = intval(date('Y'));
$sessions = [];
for ($i = 0; $i < 10; $i++) {
    $startYear = $currentYear - $i;
    $endYear = $startYear + 1;
    $sessions[] = "$startYear/" . $endYear;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Quick Course Registration with RPMS Sync</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
        }
        .card {
            background: #fff;
            border-radius: 12px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            overflow: hidden;
        }
        .card-header {
            background: linear-gradient(135deg, #0d47a1 0%, #1565c0 100%);
            color: #fff;
            padding: 25px;
            text-align: center;
        }
        .card-header h1 {
            font-size: 24px;
            margin-bottom: 5px;
        }
        .card-header p {
            opacity: 0.9;
            font-size: 14px;
        }
        .card-body {
            padding: 30px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        .form-group label {
            display: block;
            font-weight: 600;
            margin-bottom: 8px;
            color: #333;
        }
        .form-control {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 15px;
            transition: border-color 0.3s, box-shadow 0.3s;
        }
        .form-control:focus {
            outline: none;
            border-color: #1565c0;
            box-shadow: 0 0 0 3px rgba(21, 101, 192, 0.1);
        }
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
        }
        .btn {
            display: inline-block;
            padding: 14px 30px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            text-decoration: none;
        }
        .btn-primary {
            background: linear-gradient(135deg, #0d47a1 0%, #1565c0 100%);
            color: #fff;
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(13, 71, 161, 0.4);
        }
        .btn-success {
            background: linear-gradient(135deg, #2e7d32 0%, #43a047 100%);
            color: #fff;
        }
        .btn-success:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(46, 125, 50, 0.4);
        }
        .btn-secondary {
            background: #6c757d;
            color: #fff;
        }
        .btn-group {
            display: flex;
            gap: 15px;
            margin-top: 20px;
        }
        .alert {
            padding: 15px 20px;
            border-radius: 8px;
            margin-bottom: 20px;
            font-weight: 500;
        }
        .alert-success {
            background: #e8f5e9;
            color: #2e7d32;
            border: 1px solid #a5d6a7;
        }
        .alert-error {
            background: #ffebee;
            color: #c62828;
            border: 1px solid #ef9a9a;
        }
        .alert-warning {
            background: #fff3e0;
            color: #e65100;
            border: 1px solid #ffcc80;
        }
        .student-info {
            background: #f5f5f5;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 25px;
        }
        .student-info h3 {
            color: #1565c0;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 2px solid #e0e0e0;
        }
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 12px;
        }
        .info-item {
            display: flex;
        }
        .info-item strong {
            min-width: 120px;
            color: #666;
        }
        .curriculum-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 20px;
        }
        .curriculum-table th,
        .curriculum-table td {
            padding: 12px;
            text-align: left;
            border-bottom: 1px solid #e0e0e0;
        }
        .curriculum-table th {
            background: #f5f5f5;
            font-weight: 600;
            color: #333;
        }
        .curriculum-table tr:hover {
            background: #fafafa;
        }
        .status-badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        .status-c {
            background: #e3f2fd;
            color: #1565c0;
        }
        .status-e {
            background: #fff3e0;
            color: #e65100;
        }
        .status-r {
            background: #fce4ec;
            color: #c2185b;
        }
        .total-row {
            font-weight: 700;
            background: #e8f5e9 !important;
        }
        .rpms-approved {
            color: #2e7d32;
            font-weight: bold;
        }
        .rpms-pending {
            color: #f57c00;
        }
        .rpms-rejected {
            color: #c62828;
        }

        /* Print Styles */
        .print-view {
            background: #fff;
            padding: 30px;
        }
        .print-header {
            text-align: center;
            margin-bottom: 30px;
        }
        .print-header img {
            max-width: 100%;
            height: 100px;
        }
        .print-title {
            font-size: 18px;
            font-weight: 700;
            color: #333;
            margin-top: 15px;
        }
        .print-student-info {
            display: flex;
            gap: 20px;
            margin-bottom: 30px;
            border: 1px solid #ddd;
            padding: 20px;
        }
        .passport-placeholder {
            width: 150px;
            height: 150px;
            background: #f0f0f0;
            border: 2px solid #ddd;
            display: flex;
            align-items: center;
            justify-content: center;
            color: #999;
            font-size: 12px;
        }
        .passport-placeholder img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        .print-details {
            flex: 1;
        }
        .print-details table {
            width: 100%;
        }
        .print-details td {
            padding: 5px 10px;
        }
        .print-details td:first-child {
            font-weight: 600;
            width: 150px;
            text-align: right;
        }
        .signature-section {
            margin-top: 40px;
            display: flex;
            justify-content: space-between;
        }
        .signature-line {
            width: 200px;
            border-top: 1px solid #333;
            padding-top: 5px;
            text-align: center;
        }
        .rpms-badge {
            display: inline-block;
            padding: 2px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: bold;
        }
        .rpms-badge.approved {
            background: #c8e6c9;
            color: #2e7d32;
        }
        .rpms-badge.pending {
            background: #fff3e0;
            color: #e65100;
        }
        .rpms-badge.rejected {
            background: #ffcdd2;
            color: #c62828;
        }

        @media print {
            body {
                background: #fff;
                padding: 0;
            }
            .no-print {
                display: none !important;
            }
            .card {
                box-shadow: none;
            }
            .print-view {
                padding: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="card">
            <?php if ($showForm): ?>
            <div class="card-header">
                <h1>Quick Course Registration</h1>
                <p>Enter student details to auto-register core courses with RPMS sync</p>
            </div>
            <div class="card-body">
                <?php if ($message): ?>
                <div class="alert alert-<?php echo $messageType; ?>">
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <?php endif; ?>

                <form method="POST" id="regForm">
                    <div class="form-group">
                        <label for="regnum">Matric Number / JAMB Number</label>
                        <input type="text" class="form-control" id="regnum" name="regnum"
                               value="<?php echo htmlspecialchars($_POST['regnum'] ?? ''); ?>"
                               placeholder="Enter matric number (e.g., SMS/13/14/0001)" required>
                    </div>

                    <div class="form-row">
                        <div class="form-group">
                            <label for="session">Session</label>
                            <select class="form-control" id="session" name="session" required>
                                <option value="">Select Session</option>
                                <?php foreach ($sessions as $sess): ?>
                                <option value="<?php echo $sess; ?>" <?php echo (($_POST['session'] ?? '') === $sess) ? 'selected' : ''; ?>>
                                    <?php echo $sess; ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="semester">Semester</label>
                            <select class="form-control" id="semester" name="semester" required>
                                <option value="">Select Semester</option>
                                <option value="HARMATTAN" <?php echo (($_POST['semester'] ?? '') === 'HARMATTAN') ? 'selected' : ''; ?>>HARMATTAN</option>
                                <option value="RAIN" <?php echo (($_POST['semester'] ?? '') === 'RAIN') ? 'selected' : ''; ?>>RAIN</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label for="level">Level</label>
                            <select class="form-control" id="level" name="level" required>
                                <option value="">Select Level</option>
                                <?php foreach ([100, 200, 300, 400, 500, 600] as $lv): ?>
                                <option value="<?php echo $lv; ?>" <?php echo (intval($_POST['level'] ?? 0) === $lv) ? 'selected' : ''; ?>>
                                    <?php echo $lv; ?> Level
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                    </div>

                    <input type="hidden" name="action" id="formAction" value="search">

                    <div class="btn-group">
                        <button type="submit" class="btn btn-primary" onclick="document.getElementById('formAction').value='search'">
                            Search Student & Curriculum
                        </button>
                        <?php if ($student && !empty($curriculum)): ?>
                        <button type="submit" class="btn btn-success" onclick="document.getElementById('formAction').value='register'">
                            Register Courses & Sync to RPMS
                        </button>
                        <?php endif; ?>
                    </div>
                </form>

                <?php if ($student): ?>
                <div class="student-info">
                    <h3>Student Information (from Old Database)</h3>
                    <div class="info-grid">
                        <div class="info-item"><strong>Matric No:</strong> <?php echo htmlspecialchars($student['regnum']); ?></div>
                        <div class="info-item"><strong>JAMB No:</strong> <?php echo htmlspecialchars($student['jamb'] ?? 'N/A'); ?></div>
                        <div class="info-item"><strong>Name:</strong> <?php echo htmlspecialchars($student['sname'] . ' ' . $student['fname'] . ' ' . ($student['mname'] ?? '')); ?></div>
                        <div class="info-item"><strong>Programme:</strong> <?php echo htmlspecialchars($student['prog'] ?? 'N/A'); ?></div>
                        <div class="info-item"><strong>Department:</strong> <?php echo htmlspecialchars($student['dept'] ?? 'N/A'); ?></div>
                        <div class="info-item"><strong>Faculty:</strong> <?php echo htmlspecialchars($student['faculty'] ?? 'N/A'); ?></div>
                        <div class="info-item"><strong>Level:</strong> <?php echo htmlspecialchars($student['level'] ?? 'N/A'); ?></div>
                        <div class="info-item"><strong>Programme Code:</strong> <?php echo htmlspecialchars($student['pcode'] ?? 'N/A'); ?></div>
                        <div class="info-item"><strong>Status:</strong> <span style="color: green; font-weight: bold;"><?php echo htmlspecialchars($student['verif']); ?></span></div>
                    </div>
                </div>
                <?php endif; ?>

                <?php if (!empty($curriculum)): ?>
                <h3 style="margin-bottom: 15px;">Curriculum Courses (<?php echo count($curriculum); ?> courses)</h3>
                <p style="margin-bottom: 15px; color: #666;">
                    <strong>Note:</strong> Courses with status <span class="status-badge status-c">C</span> (Core) will be auto-registered and synced to RPMS.
                </p>
                <table class="curriculum-table">
                    <thead>
                        <tr>
                            <th>S/N</th>
                            <th>Code</th>
                            <th>Title</th>
                            <th>Units</th>
                            <th>Status</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $totalUnits = 0;
                        $coreUnits = 0;
                        foreach ($curriculum as $index => $course):
                            $totalUnits += intval($course['unit']);
                            if ($course['status'] === 'C') $coreUnits += intval($course['unit']);
                        ?>
                        <tr style="<?php echo $course['status'] === 'C' ? 'background: #e8f5e9;' : ''; ?>">
                            <td><?php echo $index + 1; ?></td>
                            <td><?php echo htmlspecialchars($course['code']); ?></td>
                            <td><?php echo htmlspecialchars(stripslashes($course['title'])); ?></td>
                            <td><?php echo $course['unit']; ?></td>
                            <td>
                                <span class="status-badge status-<?php echo strtolower($course['status']); ?>">
                                    <?php echo $course['status']; ?>
                                </span>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td colspan="3"><strong>Total (Core courses to be registered)</strong></td>
                            <td><strong><?php echo $coreUnits; ?></strong></td>
                            <td></td>
                        </tr>
                    </tbody>
                </table>
                <?php endif; ?>
            </div>
            <?php endif; ?>

            <?php if ($showPrint && $student && !empty($registeredCourses)): ?>
            <!-- Print View - Course Registration Form -->
            <div class="print-view">
                <div class="no-print" style="margin-bottom: 20px; text-align: center;">
                    <button onclick="window.print()" class="btn btn-primary">Print Course Form</button>
                    <a href="quick-course-registration.php" class="btn btn-secondary">Register Another Student</a>
                </div>

                <?php if ($message): ?>
                <div class="alert alert-<?php echo $messageType; ?> no-print">
                    <?php echo htmlspecialchars($message); ?>
                </div>
                <?php endif; ?>

                <div class="print-header">
                    <img src="images/newlogo2.jpg" alt="University Logo" onerror="this.style.display='none'">
                    <div class="print-title">
                        <?php echo htmlspecialchars($semester . ' ' . $session); ?> COURSE REGISTRATION FORM
                    </div>
                </div>

                <div class="print-student-info">
                    <div class="passport-placeholder">
                        <?php if (!empty($student['pass'])): ?>
                        <img src="https://portal.oouagoiwoye.edu.ng/students/<?php echo htmlspecialchars($student['pass']); ?>"
                             alt="Passport" onerror="this.parentElement.innerHTML='No Passport'">
                        <?php else: ?>
                        No Passport
                        <?php endif; ?>
                    </div>
                    <div class="print-details">
                        <table>
                            <tr>
                                <td>Matric Number:</td>
                                <td><?php echo htmlspecialchars($student['regnum']); ?></td>
                            </tr>
                            <tr>
                                <td>Name:</td>
                                <td><?php echo htmlspecialchars($student['sname'] . ' ' . $student['fname'] . ' ' . ($student['mname'] ?? '')); ?></td>
                            </tr>
                            <tr>
                                <td>Level:</td>
                                <td><?php echo $level; ?></td>
                            </tr>
                            <tr>
                                <td>Semester:</td>
                                <td><?php echo htmlspecialchars($semester); ?></td>
                            </tr>
                            <tr>
                                <td>Faculty:</td>
                                <td><?php echo htmlspecialchars($student['faculty'] ?? 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <td>Department/Programme:</td>
                                <td><?php echo htmlspecialchars(($student['dept'] ?? '') . ' (' . ($student['prog'] ?? '') . ')'); ?></td>
                            </tr>
                        </table>
                    </div>
                </div>

                <table class="curriculum-table">
                    <thead>
                        <tr>
                            <th style="width: 50px;">S/NO</th>
                            <th style="width: 100px;">CODE</th>
                            <th>COURSE TITLE</th>
                            <th style="width: 60px;">UNIT</th>
                            <th style="width: 70px;">STATUS</th>
                            <th style="width: 120px;">RPMS APPROVAL</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        $totalUnits = 0;
                        foreach ($registeredCourses as $index => $course):
                            $totalUnits += intval($course['unit']);
                            $rpmsApproved = ($course['rpms'] ?? 0) == 1;
                            $remarks = $course['remarks'] ?? '';
                        ?>
                        <tr>
                            <td style="text-align: center;"><?php echo $index + 1; ?></td>
                            <td style="text-align: center;"><?php echo htmlspecialchars($course['code']); ?></td>
                            <td><?php echo htmlspecialchars(stripslashes($course['title'])); ?></td>
                            <td style="text-align: center;"><?php echo $course['unit']; ?></td>
                            <td style="text-align: center;"><?php echo $course['status']; ?></td>
                            <td style="text-align: center;">
                                <?php if ($rpmsApproved || $remarks === 'Course Already Registered'): ?>
                                <span class="rpms-badge approved">&#10004; APPROVED</span>
                                <?php elseif (!empty($remarks)): ?>
                                <span class="rpms-badge rejected" title="<?php echo htmlspecialchars($remarks); ?>">&#10008; REJECTED</span>
                                <?php else: ?>
                                <span class="rpms-badge pending">PENDING</span>
                                <?php endif; ?>
                            </td>
                        </tr>
                        <?php if (!$rpmsApproved && !empty($remarks) && $remarks !== 'Course Already Registered'): ?>
                        <tr class="no-print">
                            <td></td>
                            <td colspan="5" style="color: #c62828; font-size: 12px; padding-top: 0;">
                                Reason: <?php echo htmlspecialchars($remarks); ?>
                            </td>
                        </tr>
                        <?php endif; ?>
                        <?php endforeach; ?>
                        <tr class="total-row">
                            <td colspan="3" style="text-align: right;"><strong>Total Number of Units Registered</strong></td>
                            <td style="text-align: center;"><strong><?php echo $totalUnits; ?></strong></td>
                            <td colspan="2"></td>
                        </tr>
                    </tbody>
                </table>

                <div style="margin-top: 20px;">
                    <p><strong>Date Registered:</strong> <?php echo date('Y-m-d H:i:s'); ?></p>
                </div>

                <div class="signature-section">
                    <div class="signature-line">
                        <strong>Signature</strong>
                    </div>
                    <div class="signature-line">
                        <strong>Part Adviser</strong>
                    </div>
                </div>

                <div style="margin-top: 30px; text-align: center; color: #666; font-size: 12px;">
                    <p>Records synced from old database with is_pushed = 1</p>
                    <p>RPMS Sync Status: <?php echo $rpmsSuccess ? 'All courses approved' : 'Some courses pending/rejected'; ?></p>
                </div>
            </div>
            <?php endif; ?>
        </div>
    </div>

    <script>
        // Auto-focus on matric number field
        document.addEventListener('DOMContentLoaded', function() {
            const regnumField = document.getElementById('regnum');
            if (regnumField) {
                regnumField.focus();
            }
        });
    </script>
</body>
</html>
<?php
// Close database connections
if ($old_db) $old_db->close();
if ($new_db) $new_db->close();
?>
